<?php
// Check and define validateToken function
if (!function_exists('validateToken')) {
    /**
     * Validate the authorization token from the request headers.
     * 
     * @param object $authorizationToken Object handling the token validation.
     * @param object $responseHandler Object to send responses.
     * @return bool True if the token is valid, false otherwise.
     */
    function validateToken($authorizationToken, $responseHandler)
    {
        $headers = $this->input->request_headers();
        if (isset($headers['authorization'])) {
            $decodedToken = $authorizationToken->validateToken($headers['authorization']);
            if (!$decodedToken['status']) {
                // Token expired or invalid
                sendErrorResponse('Token Expired!', REST_Controller::HTTP_OK, $responseHandler);
                return false;
            }
            return true;
        } else {
            sendErrorResponse('Authentication failed', REST_Controller::HTTP_OK, $responseHandler);
            return false;
        }
    }
}

// Check and define validateAndDecryptUID function
if (!function_exists('validateAndDecryptUID')) {
    /**
     * Validate and decrypt UID.
     * 
     * @param string $uid Encrypted UID.
     * @param object $encryptionHandler Object handling encryption/decryption.
     * @param object $responseHandler Object to send responses.
     * @return mixed Decrypted UID if valid, false otherwise.
     */
    function validateAndDecryptUID($uid, $encryptionHandler, $responseHandler)
    {
        $decryptedUID = $encryptionHandler->MasterDecrypt($uid);
        if (!validateUID($decryptedUID, $responseHandler)) {
            sendErrorResponse('Invalid User', REST_Controller::HTTP_BAD_REQUEST, $responseHandler);
            return false;
        }
        return $decryptedUID;
    }
}

// Check and define sendErrorResponse function
if (!function_exists('sendErrorResponse')) {
    /**
     * Send an error response.
     * 
     * @param string $message The error message.
     * @param int $httpCode The HTTP status code.
     * @param object $responseHandler Object to send responses.
     */
    function sendErrorResponse($message, $httpCode, $responseHandler)
    {
        $response = [
            'status' => false,
            'message' => $message
        ];
        $encrypted_response = $responseHandler->MasterEncrypt(json_encode($response));
        $responseHandler->response($encrypted_response, $httpCode);
    }
}

// Check and define validateUID function
if (!function_exists('validateUID')) {
    /**
     * Validate UID by checking if it exists in the database.
     * 
     * @param string $u_id The UID to validate.
     * @param object $userModel Object to interact with the user model.
     * @return bool True if the UID is valid, false otherwise.
     */
    function validateUID($u_id, $userModel)
    {
        $mdl_response = $userModel->validateUID($u_id);
        return $mdl_response !== null;
    }
}
